// Copyright (C) 2015, 2018 YesLogic Pty. Ltd.
// All rights reserved.

package com.princexml.wrapper;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.List;
import java.util.function.Consumer;

import com.princexml.wrapper.enums.InputType;
import com.princexml.wrapper.events.PrinceEvents;

/**
 * Implements support to convert List<InputStream> into a single PDF directly
 * using Prince, rather than having to convert them to byte[] first or having to
 * write them to disk first.
 */
public class CustomPrinceControl extends PrinceControl {

	/**
	 * Constructor for PrinceControl. Only use this to convert HTML to PDF.
	 * 
	 * @param exePath
	 *            The path of the Prince executable. (For example, this may be
	 *            <code>C:\Program&#xA0;Files\Prince\engine\bin\prince.exe</code> on
	 *            Windows or <code>/usr/bin/prince</code> on Linux).
	 * @param events
	 *            An instance of the PrinceEvents interface that will receive
	 *            error/warning messages returned from Prince.
	 */
	public CustomPrinceControl(final String exePath, final PrinceEvents events) {
		super(exePath, events);
		setInputType(InputType.HTML);
	}

	/**
	 * Convert a set of HTML files to a single PDF file.
	 * 
	 * @param htmlInputStreams
	 *            The inputstreams from which Prince will read.
	 * @param pdfOutput
	 *            The OutputStream to which Prince will write the PDF output.
	 * @return True if a PDF file was generated successfully.
	 */
	public boolean convertListOfInputStreams(final List<InputStream> htmlInputStreams, final OutputStream pdfOutput)
			throws IOException {
		final Consumer<OutputStream> writeChunksToPrinceFunction = (inputToPrince) -> writeBatchJobChunkToPrince(
				htmlInputStreams, inputToPrince);
		return convertWithPrince(writeChunksToPrinceFunction, pdfOutput);
	}

	// code copied mostly from PrinceControl. The only difference is we will write
	// dat-chunks
	// and supply Prince with the inputstreams the user called us with, see
	// writeBatchJobChunkToPrince()
	private boolean convertWithPrince(final Consumer<OutputStream> writeChunksToPrinceFunction,
			final OutputStream pdfOutput) throws IOException {
		if (process == null) {
			throw new RuntimeException("control process has not been started");
		}

		final OutputStream inputToPrince = process.getOutputStream();
		final InputStream outputFromPrince = process.getInputStream();

		writeChunksToPrinceFunction.accept(inputToPrince);

		inputToPrince.flush();

		Chunk chunk = Chunk.readChunk(outputFromPrince);

		if (chunk.getTag().equals("pdf")) {
			pdfOutput.write(chunk.getBytes());

			chunk = Chunk.readChunk(outputFromPrince);
		}

		if (chunk.getTag().equals("log")) {
			try (BufferedReader br = new BufferedReader(
					new InputStreamReader(new ByteArrayInputStream(chunk.getBytes())))) {
				boolean readMessages = readMessages(br);
				return readMessages;
			}
		} else if (chunk.getTag().equals("err")) {
			throw new IOException("error: " + chunk.getString());
		} else {
			throw new IOException("unknown chunk: " + chunk.getTag());
		}
	}

	// code copied mostly from PrinceControl. The only difference is we will write
	// dat-chunks
	// and supply Prince with the inputstreams the user called us with, see
	// writeBatchJobChunkToPrince()
	private String getJobJSON(final List<InputStream> inputFiles) {
		final Json json = new Json();

		json.beginObj();
		json.beginObj("input");

		if (inputFiles != null && !inputFiles.isEmpty()) {
			json.beginList("src");
			for (int i = 0; i < inputFiles.size(); i++) {
				json.value("job-resource:" + i);
			}
			json.endList();
		}

		if (inputType != null) {
			json.field("type", inputType.toString());
		}
		if (baseUrl != null) {
			json.field("base", baseUrl);
		}
		if (media != null) {
			json.field("media", media);
		}

		json.beginList("styles");
		styleSheets.forEach(json::value);
		json.endList();

		json.beginList("scripts");
		scripts.forEach(json::value);
		json.endList();

		json.field("default-style", !noDefaultStyle);
		json.field("author-style", !noAuthorStyle);
		json.field("javascript", javaScript);
		if (maxPasses > 0) {
			json.field("max-passes", maxPasses);
		}
		json.field("iframes", iframes);
		json.field("xinclude", xInclude);
		json.field("xml-external-entities", xmlExternalEntities);
		json.endObj();

		json.beginObj("pdf");
		json.field("embed-fonts", !noEmbedFonts);
		json.field("subset-fonts", !noSubsetFonts);
		json.field("artificial-fonts", !noArtificialFonts);
		json.field("force-identity-encoding", forceIdentityEncoding);
		json.field("compress", !noCompress);
		json.field("object-streams", !noObjectStreams);

		json.beginObj("encrypt");
		if (keyBits != null) {
			json.field("key-bits", keyBits.getValue());
		}
		if (userPassword != null) {
			json.field("user-password", userPassword);
		}
		if (ownerPassword != null) {
			json.field("owner-password", ownerPassword);
		}
		json.field("disallow-print", disallowPrint);
		json.field("disallow-modify", disallowModify);
		json.field("disallow-copy", disallowCopy);
		json.field("disallow-annotate", disallowAnnotate);
		json.field("allow-copy-for-accessibility", allowCopyForAccessibility);
		json.field("allow-assembly", allowAssembly);
		json.endObj();

		if (pdfProfile != null) {
			json.field("pdf-profile", pdfProfile.toString());
		}
		if (pdfOutputIntent != null) {
			json.field("pdf-output-intent", pdfOutputIntent);
		}
		if (pdfScript != null) {
			json.beginObj("pdf-script");
			json.field("url", pdfScript);
			json.endObj();
		}
		if (!pdfEventScripts.isEmpty()) {
			json.beginObj("pdf-event-scripts");
			pdfEventScripts.forEach((k, v) -> {
				json.beginObj(k.toString());
				json.field("url", v);
				json.endObj();
			});
			json.endObj();
		}
		if (fallbackCmykProfile != null) {
			json.field("fallback-cmyk-profile", fallbackCmykProfile);
		}
		json.field("color-conversion", convertColors ? "output-intent" : "none");
		if (pdfId != null) {
			json.field("pdf-id", pdfId);
		}
		if (pdfLang != null) {
			json.field("pdf-lang", pdfLang);
		}
		if (xmp != null) {
			json.field("pdf-xmp", xmp);
		}
		json.field("tagged-pdf", taggedPdf);
		json.field("pdf-forms", pdfForms);

		json.beginList("attach");
		for (FileAttachment fa : fileAttachments) {
			json.beginObj();
			json.field("url", fa.url);
			if (fa.filename != null) {
				json.field("filename", fa.filename);
			}
			if (fa.description != null) {
				json.field("description", fa.description);
			}
			json.endObj();
		}
		json.endList();

		json.endObj();

		json.beginObj("metadata");
		if (pdfTitle != null) {
			json.field("title", pdfTitle);
		}
		if (pdfSubject != null) {
			json.field("subject", pdfSubject);
		}
		if (pdfAuthor != null) {
			json.field("author", pdfAuthor);
		}
		if (pdfKeywords != null) {
			json.field("keywords", pdfKeywords);
		}
		if (pdfCreator != null) {
			json.field("creator", pdfCreator);
		}
		json.endObj();
		if (inputFiles != null && !inputFiles.isEmpty()) {
			json.field("job-resource-count", inputFiles.size());
		}
		json.endObj();
		return json.toString();
	}

	private void writeBatchJobChunkToPrince(final List<InputStream> xmlInputStreams, final OutputStream inputToPrince) {
		try {
			Chunk.writeChunk(inputToPrince, "job", getJobJSON(xmlInputStreams));
			for (final InputStream inputStream : xmlInputStreams) {
				Chunk.writeChunk(inputToPrince, "dat", inputStream);
			}
		} catch (final IOException e) {
			throw new RuntimeException(e);
		}
	}
}
